import ctypes
import json
import os
import queue
import subprocess
import threading
import time
from typing import Dict, List

import psutil
from flask import Flask, jsonify, request

try:
    import win32gui
    import win32con
    import win32process
except Exception:
    win32gui = None

APP = Flask(__name__)

TOKEN = os.getenv("AGENT_TOKEN", "change_me_agent_token")
HOST = "127.0.0.1"
PORT = 17171

SSH_TUNNEL_ENABLED = os.getenv("SSH_TUNNEL_ENABLED", "true").lower() in {"1", "true", "yes", "on"}
SSH_USER = os.getenv("SSH_USER", "ubuntu")
SSH_HOST = os.getenv("SSH_HOST", "remote.thais.ovh")
SSH_PORT = os.getenv("SSH_PORT", "2222")
SSH_KEY_PATH = os.getenv("SSH_KEY_PATH", "")
REVERSE_VNC_PORT = os.getenv("REVERSE_VNC_PORT", "15900")
REVERSE_AGENT_PORT = os.getenv("REVERSE_AGENT_PORT", "17171")

_state = {
    "connected": False,
    "locked": False,
    "allowed_apps": [],
    "forbidden_apps": [],
}

_events = queue.Queue()


def _auth() -> bool:
    return request.headers.get("X-Agent-Token") == TOKEN


def _block_input(enabled: bool) -> None:
    try:
        ctypes.windll.user32.BlockInput(1 if enabled else 0)
        _state["locked"] = enabled
    except Exception:
        pass


def _kill_forbidden_apps() -> List[str]:
    killed = []
    forbidden = set(a.lower() for a in _state.get("forbidden_apps", []))
    if not forbidden:
        return killed
    for proc in psutil.process_iter(["name", "exe"]):
        try:
            name = (proc.info.get("name") or "").lower()
            exe = (proc.info.get("exe") or "").lower()
            if name in forbidden or exe in forbidden:
                proc.kill()
                killed.append(name or exe)
        except Exception:
            continue
    return killed


def _focus_allowed_app() -> bool:
    allowed = _state.get("allowed_apps", [])
    if not allowed or win32gui is None:
        return False
    allowed = [a.lower() for a in allowed]

    def _enum_handler(hwnd, result):
        if not win32gui.IsWindowVisible(hwnd):
            return
        _, pid = win32process.GetWindowThreadProcessId(hwnd)
        try:
            proc = psutil.Process(pid)
            name = proc.name().lower()
            exe = (proc.exe() or "").lower()
            if name in allowed or exe in allowed:
                result.append(hwnd)
        except Exception:
            return

    windows = []
    win32gui.EnumWindows(_enum_handler, windows)
    if not windows:
        return False
    hwnd = windows[0]
    win32gui.ShowWindow(hwnd, win32con.SW_RESTORE)
    win32gui.SetForegroundWindow(hwnd)
    return True


class OverlayUI(threading.Thread):
    def __init__(self) -> None:
        super().__init__(daemon=True)
        self._connected = False

    def run(self) -> None:
        try:
            import tkinter as tk
        except Exception:
            return

        root = tk.Tk()
        root.title("Remote Status")
        root.attributes("-topmost", True)
        root.geometry("48x48+20+20")
        root.configure(bg="#0b0e13")
        root.overrideredirect(True)

        canvas = tk.Canvas(root, width=48, height=48, highlightthickness=0, bg="#0b0e13")
        canvas.pack()
        dot = canvas.create_oval(10, 10, 38, 38, fill="#ff5c5c", outline="#0b0e13")

        def tick():
            while not _events.empty():
                event, payload = _events.get_nowait()
                if event == "connected":
                    self._connected = bool(payload)
            color = "#3ee6a3" if self._connected else "#ff5c5c"
            canvas.itemconfig(dot, fill=color)
            root.after(500, tick)

        tick()
        root.mainloop()


class TunnelWorker(threading.Thread):
    def __init__(self) -> None:
        super().__init__(daemon=True)
        self._proc = None

    def _cmd(self) -> List[str]:
        cmd = [
            "ssh",
            "-N",
            "-o",
            "ServerAliveInterval=30",
            "-o",
            "ServerAliveCountMax=3",
            "-o",
            "ExitOnForwardFailure=yes",
            "-o",
            "StrictHostKeyChecking=accept-new",
            "-p",
            str(SSH_PORT),
            "-R",
            f"{REVERSE_VNC_PORT}:127.0.0.1:5900",
            "-R",
            f"{REVERSE_AGENT_PORT}:127.0.0.1:17171",
        ]
        if SSH_KEY_PATH:
            cmd.extend(["-i", SSH_KEY_PATH])
        cmd.append(f"{SSH_USER}@{SSH_HOST}")
        return cmd

    def run(self) -> None:
        if not SSH_TUNNEL_ENABLED:
            return
        creation_flags = 0
        if os.name == "nt":
            creation_flags = subprocess.CREATE_NO_WINDOW
        while True:
            try:
                self._proc = subprocess.Popen(self._cmd(), creationflags=creation_flags)
                self._proc.wait()
            except Exception:
                pass
            time.sleep(5)


overlay = OverlayUI()
tunnel = TunnelWorker()


@APP.before_request
def _check_auth():
    if request.path == "/health":
        return
    if not _auth():
        return jsonify({"error": "unauthorized"}), 401


@APP.get("/health")
def health():
    return {"ok": True}


@APP.post("/status/connected")
def status_connected():
    data = request.get_json(force=True)
    connected = bool(data.get("connected"))
    _state["connected"] = connected
    _events.put(("connected", connected))
    return {"connected": connected}


@APP.post("/input/lock")
def input_lock():
    data = request.get_json(force=True)
    enabled = bool(data.get("enabled"))
    _block_input(enabled)
    return {"locked": enabled}


@APP.post("/kiosk/enforce")
def kiosk_enforce():
    data = request.get_json(force=True)
    _state["allowed_apps"] = data.get("allowed_apps", [])
    _state["forbidden_apps"] = data.get("forbidden_apps", [])
    return {"ok": True}


@APP.post("/kiosk/kill-forbidden")
def kiosk_kill():
    killed = _kill_forbidden_apps()
    return {"killed": killed}


@APP.post("/kiosk/focus")
def kiosk_focus():
    ok = _focus_allowed_app()
    return {"focused": ok}


@APP.post("/overlay/message")
def overlay_message():
    data = request.get_json(force=True)
    message = data.get("message", "")
    return {"ok": True, "message": message}


def main() -> None:
    overlay.start()
    tunnel.start()
    APP.run(host=HOST, port=PORT)


if __name__ == "__main__":
    main()
